<?php
/**
 * @since      1.0.0
 *
 * @package    Ftc_Instagram_Feed_Gallery_Widget
 * @subpackage Ftc_Instagram_Feed_Gallery_Widget/includes
 */

/**
 * The Social Gallery and Widget Widget for the plugin.
 *
 * @package    Ftc_Instagram_Feed_Gallery_Widget
 * @subpackage Ftc_Instagram_Feed_Gallery_Widget/includes
 *
 */
class Ftc_Instagram_Feed_Gallery_Widget_Widget extends WP_Widget {

	/**
	 * Holds widget settings defaults, populated in constructor.
	 *
	 * @var array
	 */
	protected $defaults;

	/**
	 * Holds widget settings options, populated in constructor.
	 *
	 * @var array
	 */
	protected $options;

	/**
	 * Constructor. Set the default widget options and create widget.
	 *
	 * @since 1.0.0
	 */
	function __construct() {
		$this->defaults = array(
			'title'     => esc_html__( 'FTC Instagram Feed Widget', 'ftc-instagram-feed-gallery-widget' ),
			'type'      => 'users',
			'username'  => '',
			'photos_number'    => 6,
			'padding'   => 0,
			'layout'    => 'default',
			'columns'    => 'six-columns',
			'size'      => 'standard_resolution',
			'target'    => 0,
			'link_type' => 'link_site',
			'link_text' => esc_html__( 'View on Instagram', 'ftc-instagram-feed-gallery-widget' ),
		);

		$widget_ops = array(
			'classname'   => 'ftc-instagram-feed-gallery-widget',
			'description' => esc_html__( 'Displays your latest Instagram feed as Gallery and Widget', 'ftc-instagram-feed-gallery-widget' ),
		);

		$control_ops = array(
			'id_base' => 'ftc-instagram-feed-gallery-widget',
		);

		parent::__construct(
			'ftc-instagram-feed-gallery-widget', // Base ID
			esc_html__( 'FTC Instagram Feed Widget', 'ftc-instagram-feed-gallery-widget' ), // Name.
			$widget_ops,
			$control_ops
		);
	}

	/**
	 * Widget Form. Create widget form.
	 *
	 * @since 1.0.0
	 *
	 * @param array $instance Widget Instance.
	 */
	function form( $instance ) {
		$instance = wp_parse_args( (array) $instance, $this->defaults );

		$options = ftc_instagram_feed_gallery_widget_get_options( 'ftc_instagram_feed_gallery_widget_options' );
		?>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"><?php esc_html_e( 'Title', 'ftc-instagram-feed-gallery-widget' ); ?>:</label>
			<input type="text" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" value="<?php echo esc_attr( $instance['title'] ); ?>" class="widefat ftc-instagram-feed-gallery-widget-title" />
		</p>

		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'username' ) ); ?>"><?php esc_html_e( 'Username', 'ftc-instagram-feed-gallery-widget' ); ?>:</label>
			<input disabled="disabled" type="text" id="<?php echo esc_attr( $this->get_field_id( 'username' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'username' ) ); ?>" value="<?php echo esc_attr( $options['username'] ); ?>" class="widefat ftc-instagram-feed-gallery-widget-username" />
		</p>
		<?php if ( ! $options['username'] ) : ?>
		<p class="description">
			<?php
			$url = admin_url( 'options-general.php?page=ftc-instagrams-widget' );

			/* translators: 1. link start, 2. link end */
			printf(
				esc_html__( 'Please fetch access token %1$shere%2$s', 'ftc-instagram-feed-gallery-widget' ),
				'<a href="' . esc_url( $url ) . '">',
				'</a>'
			);
			?>
		</p>
		<?php endif; ?>

		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'photos_number' ) ); ?>"><?php esc_html_e( 'Number of photos', 'ftc-instagram-feed-gallery-widget' ); ?>:</label>
			<input type="number" id="<?php echo esc_attr( $this->get_field_id( 'photos_number' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'photos_number' ) ); ?>" value="<?php echo absint( $instance['photos_number'] ); ?>" class="small-text ftc-instagram-feed-gallery-widget-number" max="30" min="1" />
		</p>
		<p class="description"><?php esc_html_e( 'Max is 30.', 'ftc-instagram-feed-gallery-widget' ); ?></p>

		<?php

	}

	/**
	 * Widget Update. Save/Update widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @param array $new_instance Widget New Instance.
	 * @param array $old_instance Widget Old Instance.
	 */
	function update( $new_instance, $old_instance ) {
		$instance = $old_instance;

		$instance['title']    = sanitize_text_field( $new_instance['title'] );
		$instance['username'] = sanitize_text_field( $new_instance['username'] );
		$instance['photos_number']   = ftc_instagram_feed_gallery_widget_sanitize_number_range( $new_instance['photos_number'] );
		$instance['size']     = sanitize_key( $new_instance['size'] );
		delete_transient( 'ftc_insta_json' );

		return $instance;
	}

	/**
	 * Widget Form. Create widget form.
	 *
	 * @since 1.0.0
	 *
	 * @param array $args Arguments.
	 * @param array $instance Instance.
	 */
	function widget( $args, $instance ) {
		// Merge with defaults.
		$instance = wp_parse_args( (array) $instance, $this->defaults );

		echo $args['before_widget']; // WPCS: XSS ok, sanitization ok.

		// Set up the author bio.
		if ( ! empty( $instance['title'] ) ) {
			echo $args['before_title'] . apply_filters( 'widget_title', $instance['title'], $instance, $this->id_base ) . $args['after_title']; // WPCS: XSS ok, sanitization ok.
		}
		$instance['element'] = 'widget';

		$options  = ftc_instagram_feed_gallery_widget_get_options( 'ftc_instagram_feed_gallery_widget_options' );
		$username = $options['username'];

		if ( ! $instance['username'] ) {
			$instance['username'] = $username;
		}

		$ftc_instagram_feed_gallery_widget_helper = new Ftc_Instagram_Feed_Gallery_Widget_Helper();
		echo $ftc_instagram_feed_gallery_widget_helper->display( $instance ); // WPCS: XSS ok, sanitization ok.

		echo $args['after_widget']; // WPCS: XSS ok, sanitization ok.
	}

}

/**
 * Intiate Ftc_Instagram_Feed_Gallery_Widget_Widget Class.
 *
 * @since 1.0.0
 */
function ftc_instagram_feed_gallery_widget_register_widget() {
	register_widget( 'Ftc_Instagram_Feed_Gallery_Widget_Widget' );
}
add_action( 'widgets_init', 'ftc_instagram_feed_gallery_widget_register_widget' );
